; Cmdline.asm - Displays all command line parameters
;
; Note that because of the large amount of stack space used in the
; main procedure I tell the linker to reserve and commit 16K of
; stack space. An alternative to this would be to allocate space
; in the bss section and set this to the stack upon startup.
;

%define _WINCONSOLE_
%include "Gaz\Win32\Include\Windows.inc"

[BITS 32]
[section .text]

procglobal main
	ddlocal		_hStdIn, _hStdOut
	;
	; _argc will hold the number of command line arguments
	;
	ddstatic	_argc, _tmp
	;
	; _argv will hold pointers to each command line argument,
	; and _szargv will hold the strings themselves. I assume the
	; total length of all the strings is less than 16K and that
	; there are no more than 256 arguments... really, I should
	; either check I won't exceed these limits or dynamically
	; allocate memory for them instead.
	;
	; Turning the command line parser into a DLL function would
	; be a great exercise, using memory allocation for the arguments.
	; A nice feature would be the function taking a parameter - the
	; argument separator.
	;
	bufflocal	_argv, 256*4, _szargv, 4096*4, _szBuffer, 256, _szBuffer2, 256
	endlocals
	;
	TEXTlocal _szTitle, 'Cmdline',0
	sc SetConsoleTitle, ._szTitle
	;
	sc GetStdHandle, STD_INPUT_HANDLE
	mov	._hStdIn, eax
	sc GetStdHandle, STD_OUTPUT_HANDLE
	mov	._hStdOut, eax
	;
	; We can easily obtain a pointer to the command line, and the API
	; has a call to parse it automatically, but it only works with Unicode
	; strings. Hence, we need to parse it ourselves here. The strategy is
	; to loop through the whole string checking each character and generating
	; an array of pointers to the arguments, and we copy the arguments to a
	; buffer as dword-aligned null-terminated strings (as this is convenient
	; for all string operations in Win32)
	;
	; This parser isn't a particularly good one - really it should know
	; about using quotes to surround arguments which include spaces.
	;
	sc GetCommandLine			; get pointer to the command line
	mov	._argc, dword -1		; _argc holds the argument count, starting at -1
						; because the first paramter is the program itself
	mov	esi, ._argv			; esi points to the argument pointer list
	mov	edi, ._szargv			; edi points to the argument buffer itself
	mov	[esi], edi			; first argument points to the start of the buffer
	add	esi, 4				; onto the next argument pointer
	while [eax], ne, byte 0			; loop until the end of the string is found
		if [eax], e, byte ' '		; argument delimiter? (space)
			while [eax], e, byte ' '
				inc	eax	; ignore any multiple spaces
			wend
			cmp	[eax], byte 0	; end of command line? (if no arguments this is true)
			jz	_cmdline_end	; yes, end the search here
			mov	[edi], byte 0	; no, add a null terminator to the string
			inc	edi		; onto the next byte in the buffer
			mov	ebx, edi	; get the address so we can align to a dword boundary
			and	ebx, 3		; mask off all but bottom two bits
			sub	ebx, 4		; subtract 4, so 4 = 0, 3 = -1, 2 = -2, 1 = -3, 0 = -4
			and	ebx, 3		; and mask off all but bottom two bits to get 0, 1, 2, 3 or 4
			add	edi, ebx	; and add to the buffer address to dword align it
			mov	[esi], edi	; set the pointer for the next argument
			add	esi, 4		; onto the next argument pointer
			inc	._argc		; increase the argument count
		else
			mov	bl, [eax]	; no, copy byte to buffer
			mov	[edi], bl
			inc	eax		; onto the next character in the command line
			inc	edi		; and in the buffer
		endif
	wend
_cmdline_end:
	inc	._argc				; add in the final argument
	mov	[esi], dword 0			; add an end of argument list marker
	;
	; Now we've parsed the command line we display a string to tell the
	; user the number of parameters that were passed
	;
	TEXTlocal _szOutput, "%d paramters",13,10,0
	sc wvsprintf, ._szBuffer, ._szOutput, .._argc
	;
	; eax now holds the number of characters written
	;
	sc WriteConsole, ._hStdOut, ._szBuffer, eax, .._argc, NULL
	;
	; With the number displayed, we loop through the array of pointers
	; to each argument and display each one followed by a newline string
	; so it's one argument / line. Don't forget that there is always a
	; parameter 0 - the program itself
	;
	mov	esi, ._argv			; esi points to the pointer array
	while [esi], ne, dword 0		; loop until the end of array marker is found
		;
		; copy the string from the array buffer to a local buffer, so
		; we can get the number of characters in the string
		;
		sc wvsprintf, ._szBuffer2, [esi], 0
		;
		; display the argument string
		;
		sc WriteConsole, ._hStdOut, ._szBuffer2, eax, .._tmp, NULL
		;
		; output a newline string
		;
		TEXTlocal _szNextLine, 10,13,0
		sc WriteConsole, ._hStdOut, ._szNextLine, 2, .._tmp, NULL
		add	esi,4			; next pointer in the array
	wend
	;
	; exit the application
	;
	sc ExitProcess, 0
endproc

[section .bss]

[section .data]
